package com.tencent.start.cgs.tools;

import com.emc.ecs.nfsclient.nfs.*;
import com.emc.ecs.nfsclient.nfs.io.Nfs3File;
import com.emc.ecs.nfsclient.nfs.io.NfsFile;

import java.io.*;
import java.nio.ByteBuffer;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.atomic.AtomicInteger;

import static com.emc.ecs.nfsclient.nfs.NfsWriteRequest.DATA_SYNC;

@SuppressWarnings("unused")
public abstract class RandomAccessOutputFile extends RandomAccessDataOutput {

    public abstract long length() throws IOException;
    public abstract long getFilePointer() throws IOException;
    public abstract void seek(long pos) throws IOException;
    public abstract void setLength(long newLength) throws IOException;

    public static class WrapperRandomAccessOutputFile extends RandomAccessOutputFile {
        public final RandomAccessOutputFile output;

        public WrapperRandomAccessOutputFile(RandomAccessOutputFile out) {
            this.output = out;
        }

        @Override
        public long length() throws IOException {
            return output.length();
        }

        @Override
        public long getFilePointer() throws IOException {
            return output.getFilePointer();
        }

        @Override
        public void seek(long pos) throws IOException {
            output.seek(pos);
        }

        @Override
        public void setLength(long newLength) throws IOException {
            output.setLength(newLength);
        }

        @Override
        public void flush() throws IOException {
            output.flush();
        }

        @Override
        public void close() throws IOException {
            output.close();
        }

        @Override
        public void write(int b) throws IOException {
            output.write(b);
        }

        @Override
        public void write(byte[] b, int off, int len) throws IOException {
            output.write(b, off, len);
        }
    }

    public static class RandomAccessNativeOutputFile extends RandomAccessOutputFile {
        final RandomAccessFile file;

        public RandomAccessNativeOutputFile(RandomAccessFile file) {
            this.file = file;
        }

        public RandomAccessNativeOutputFile(File file) throws IOException {
            this.file = new RandomAccessFile(file, "rw");
        }

        @Override
        public void close() throws IOException {
            file.close();
        }

        @Override
        public long length() throws IOException {
            return file.length();
        }

        @Override
        public long getFilePointer() throws IOException {
            return file.getFilePointer();
        }

        @Override
        public void seek(long pos) throws IOException {
            file.seek(pos);
        }

        @Override
        public void setLength(long newLength) throws IOException {
            file.setLength(newLength);
        }

        @Override
        public void flush() throws IOException {
            file.getFD().sync();
        }

        @Override
        public void write(int b) throws IOException {
            file.write(b);
        }

        @Override
        public void write(byte[] b, int off, int len) throws IOException {
            file.write(b, off, len);
        }
    }

    public static RandomAccessOutputFile from(RandomAccessFile file) {
        return new RandomAccessNativeOutputFile(file);
    }

    public static RandomAccessOutputFile from(File file) throws IOException {
        return new RandomAccessNativeOutputFile(new RandomAccessFile(file, "rw"));
    }

    static int createNewFile(Nfs3File file) {
        try {
            NfsSetAttributes attributes = new NfsSetAttributes(
                    NfsFile.ownerReadModeBit | NfsFile.groupReadModeBit | NfsFile.othersReadModeBit,
                    0L, 0L, NfsTime.SET_TO_CURRENT_ON_SERVER, NfsTime.SET_TO_CURRENT_ON_SERVER);
            file.create(NfsCreateMode.GUARDED, attributes, null);
        } catch (NfsException e) {
            if (e.getStatus() == NfsStatus.NFS3ERR_EXIST) {
                return 0;
            }
            return -1;
        } catch (IOException e) {
            e.printStackTrace();
            return -2;
        }
        return 1;
    }

    static final int MaxWriteBytes = 1048576;

    public static class RandomAccessNfsOutputFile extends RandomAccessOutputFile {
        private final Nfs3File file;
        private final ByteBuffer buf;
        private long filePos = 0;
        private boolean created = false;

        public RandomAccessNfsOutputFile(Nfs3File file) {
            this(file, 65536);
        }

        public RandomAccessNfsOutputFile(Nfs3File file, int bufSize) {
            this.file = file;
            this.buf = ByteBuffer.allocate(bufSize);
        }

        @Override
        public void close() {
        }

        @Override
        public long length() throws IOException {
            return file.lengthEx();
        }

        @Override
        public long getFilePointer() {
            return filePos + buf.position();
        }

        @Override
        public void seek(long pos) throws IOException {
            if (pos < 0) {
                throw new IOException("Negative seek offset");
            } else if (pos != getFilePointer()) {
                sendImmediately();
                filePos = pos;
            }
        }

        @Override
        public void setLength(long newLength) throws IOException {
            NfsSetAttributes attr = new NfsSetAttributes();
            attr.setSize(newLength);
            file.setAttributes(attr);
        }

        @Override
        public void flush() throws IOException {
            sendImmediately();
        }

        @Override
        public void write(int b) throws IOException {
            if (0 == buf.remaining()) {
                sendImmediately();
            }
            buf.put((byte) b);
        }

        @Override
        public void write(byte[] b, int off, int len) throws IOException {
            if (off < 0 || len < 0 || off > len) {
                throw new IOException("Negative write offset");
            }
            do {
                int size = buf.remaining();
                if (0 == size) {
                    sendImmediately();
                    size = buf.remaining();
                }
                if (size > len - off) {
                    size = len - off;
                }
                buf.put(b, off, size);
                off += size;
            } while (off < len);
        }

        public void sendImmediately() throws IOException {
            if (buf.position() <= 0) {
                buf.clear();
                return;
            }
            if (!created && !(created = (createNewFile(file) >= 0))) {
                throw new IOException("count not create file: " + file);
            }
            long pos = filePos;
            filePos += buf.position();
            buf.flip();
            // Collections.singletonList(buf) will throw exception
            // Nfs3 will add element
            List<ByteBuffer> payloads = new ArrayList<>(2);
            payloads.add(buf);
            NfsWriteResponse resp = file.write(pos, payloads, DATA_SYNC);
            if (!resp.stateIsOk()) {
                throw new IOException(resp.toString());
            }
        }
    }

    public static class RandomAccessOutputNfsFileAsync extends RandomAccessOutputFile {
        private final Nfs3File file;
        private final int bufSize;
        private boolean created = false;
        private boolean closed = false;
        private long filePos = 0;
        private Buffer writeBuffer;
        private Buffer writeQueueHead = null;
        private Buffer writeQueueTail = null;
        private final Object writeQueueLock = new Object();
        private final AtomicInteger queueSize = new AtomicInteger(0);
        private final int maxQueueSize;
        private final Thread writeThread;
        private IOException exception;

        private static class Buffer {
            Buffer next;
            final ByteBuffer buf;
            long pos = 0;

            Buffer(int bufSize) {
                buf = ByteBuffer.allocate(bufSize);
            }
        }

        public RandomAccessOutputNfsFileAsync(Nfs3File file) {
            this(file, 65536, 1024);
        }

        public RandomAccessOutputNfsFileAsync(Nfs3File file, int bufSize, int maxQueueSize) {
            this.file = file;
            this.bufSize = bufSize;
            this.maxQueueSize = maxQueueSize;
            this.writeBuffer = new Buffer(bufSize);
            writeThread = new Thread(() -> {
                while (exception == null) {
                    Buffer writeQueue;
                    int queueSize;
                    synchronized (writeQueueLock) {
                        if (writeQueueHead == null && !closed) {
                            try {
                                writeQueueLock.wait();
                            } catch (InterruptedException e) {
                                exception = new IOException(e.getMessage());
                                return;
                            }
                        }
                        writeQueue = writeQueueHead;
                        writeQueueHead = writeQueueTail = null;
                        queueSize = this.queueSize.getAndSet(0);
                    }
                    if (queueSize >= maxQueueSize) {
                        synchronized (this.queueSize) {
                            this.queueSize.notify();
                        }
                    }
                    if (writeQueue != null) {
                        nfsWrite(writeQueue);
                    } else if (closed) {
                        break;
                    }
                }
            });
            writeThread.start();
        }

        private void nfsWrite(Buffer writeQueue) {
            try {
                if (!created && !(created = (createNewFile(file) >= 0))) {
                    throw new IOException("count not create file: " + file);
                }
                final int maxPayloadSize = MaxWriteBytes - 4096;
                Buffer next = writeQueue;
                while(next != null) {
                    Buffer buffer = next;
                    next = next.next;
                    // Collections.singletonList(buf) will throw exception
                    // Nfs3 will add element
                    List<ByteBuffer> payloads = new ArrayList<>((MaxWriteBytes + bufSize - 1) / bufSize);
                    long pos = buffer.pos;
                    long payloadSize = buffer.buf.remaining();
                    payloads.add(buffer .buf);

                    while (next != null) {
                        if (payloadSize + next.buf.remaining() > maxPayloadSize) {
                            break;
                        } else if (pos + payloadSize == next.pos) {
                            payloads.add(next.buf);
                        } else if (next.pos + next.buf.remaining() == pos) {
                            payloads.add(0, next.buf);
                            pos = next.pos;
                        } else {
                            break;
                        }
                        payloadSize += next.buf.remaining();
                        next = next.next;
                    }

                    NfsWriteResponse resp = file.write(pos, payloads, DATA_SYNC);
                    if (!resp.stateIsOk()) {
                        throw new IOException(resp.toString());
                    }
                }
            } catch (IOException e) {
                exception = e;
            }
        }

        @Override
        public void close() throws IOException {
            if (exception != null) {
                throw exception;
            }
            closed = true;
            if (null != writeThread) {
                try {
                    synchronized (writeQueueLock) {
                        writeQueueLock.notify();
                    }
                    writeThread.join();
                } catch (InterruptedException e) {
                    e.printStackTrace();
                }
            }
            Buffer buffer = flushBuffer();
            if (buffer != null) {
                queue(buffer, true);
            }
            buffer = writeQueueHead;
            writeQueueHead = writeQueueTail = null;
            queueSize.getAndSet(0);
            if (buffer != null) {
                nfsWrite(buffer);
            }
        }

        @Override
        public long length() throws IOException {
            return file.lengthEx();
        }

        @Override
        public synchronized long getFilePointer() {
            return filePos + writeBuffer.buf.position();
        }

        @Override
        public void seek(long pos) throws IOException {
            if (pos < 0) {
                throw new IOException("Negative seek offset");
            } else if (exception != null) {
                throw exception;
            } else if (pos != getFilePointer()) {
                Buffer buffer;
                synchronized (this) {
                    buffer = flushBuffer();
                    filePos = pos;
                }
                if (buffer != null) {
                    queue(buffer, false);
                }
            }
        }

        @Override
        public void setLength(long newLength) throws IOException {
            NfsSetAttributes attr = new NfsSetAttributes();
            attr.setSize(newLength);
            file.setAttributes(attr);
        }

        @Override
        public void flush() {
        }

        @Override
        public void write(int b) throws IOException {
            if (exception != null) {
                throw exception;
            }
            Buffer buffer = null;
            synchronized (this) {
                if (0 == writeBuffer.buf.remaining()) {
                    buffer = flushBuffer();
                }
                writeBuffer.buf.put((byte) b);
            }
            if (buffer != null) {
                queue(buffer, false);
            }
        }

        @Override
        public void write(byte[] b, int off, int len) throws IOException {
            if (off < 0 || len < 0 || off > len) {
                throw new IOException("Negative write offset");
            } else if (exception != null) {
                throw exception;
            }
            do {
                Buffer buffer = null;
                synchronized (this) {
                    int size = writeBuffer.buf.remaining();
                    if (0 == size) {
                        buffer = flushBuffer();
                        size = writeBuffer.buf.remaining();
                    }
                    if (size > len - off) {
                        size = len - off;
                    }
                    writeBuffer.buf.put(b, off, size);
                    off += size;
                }
                if (buffer != null) {
                    queue(buffer, false);
                }
            } while (off < len);
        }

        private Buffer flushBuffer() {
            if (writeBuffer.buf.position() == 0) {
                writeBuffer.buf.clear();
                return null;
            }
            Buffer buffer = writeBuffer;
            buffer.pos = filePos;
            writeBuffer = new Buffer(bufSize);
            filePos += buffer.buf.position();
            buffer.buf.flip();
            return buffer;
        }

        private void queue(Buffer buffer, boolean force) throws IOException {
            for (;;) {
                synchronized (writeQueueLock) {
                    if (force || queueSize.get() < maxQueueSize) {
                        queueSize.incrementAndGet();
                        if (writeQueueTail != null) {
                            writeQueueTail.next = buffer;
                            writeQueueTail = buffer;
                        } else {
                            App.Assert(writeQueueHead == null);
                            writeQueueHead = writeQueueTail = buffer;
                        }
                        buffer.next = null;
                        writeQueueLock.notify();
                        return;
                    }
                }
                synchronized (queueSize) {
                    if (queueSize.get() >= maxQueueSize) {
                        try {
                            queueSize.wait();
                        } catch (InterruptedException e) {
                            throw new IOException(e.getMessage());
                        }
                    }
                }
            }
        }
    }

    public static RandomAccessOutputFile from(final Nfs3File file) {
        return new RandomAccessOutputNfsFileAsync(file);
    }

    public static RandomAccessOutputFile from(AbstractFile file) throws IOException {
        if (file.getClassType().equals(Nfs3File.class)) {
            return new RandomAccessOutputNfsFileAsync((Nfs3File) file.asFile());
        } else {
            return new RandomAccessNativeOutputFile((File) file.asFile());
        }
    }
}
