package com.tencent.start.cgs.tools;

/*
Header format:
    4b MAGIC:PKFL
    1b MajorVer
    1b MinorVer
    2b byte order check
    1b Header size
    1b Node size
    4b Nodes offset
    4b Nodes count
    2b padding

Node format:
    1b High 4bits flags (0x80 directory), low 4bits nameLengthHigh
    struct {
        1b nameLengthLow
        4b name ptr
    }
    2b lastModifiedHigh
    4b lastModifiedLow
    8b fileSize
    union {
        8b dataOffset
        struct {
            4b childNodesCount
            4b childNodesListPtr
        }
    }
 */

import org.apache.commons.codec.binary.Hex;

import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.nio.charset.StandardCharsets;
import java.text.SimpleDateFormat;
import java.util.*;

@SuppressWarnings("unused")
public class ZpkMetadata {
    public static final String MD_FORMAT_MAGIC = "PKFL";
    public static final int MD_MAJOR_VER = 1;
    public static final int MD_MINOR_VER = 2;
    public static final int MD_MINOR_VER_INVALID = 0;
    public static final int MD_BYTEORDER_FLAG = 255;
    public static final int MD_HEADER_SIZE = 20;
    public static final int MD_NODE_SIZE = 28;
    public static final int MD_CHECKSUM_SIZE = 16;
    /*
    private static String getFileName(AbstractFile file) {
        return getStandardName(file.getName());
    }

    private static String getStandardName(String name) {
        return name.equals(".") ? "" : name.startsWith("./") ? name.substring(2) : name;
    }
    */
    public static class FileNode implements Comparable<FileNode> {
        public String relativeName;
        public final String fileName;
        public final long lastModified;
        public long fileSize;
        public long dataOffset;
        public final List<FileNode> childNodes;
        public int totalChildNoesCount;
        public byte[] md5sum;
        public int childIndex = -1;
        public Object ud, ud2;

        @Override
        public int compareTo(FileNode o) {
            return fileName.compareToIgnoreCase(o.fileName);
        }

        public FileNode(String fileName, long lastModified, long fileSize, long dataOffset, List<FileNode> childNodes)
                throws IOException {
            if (fileName.equals(".") || fileName.startsWith("./")) {
                throw new IOException("invalid file name: " + fileName);
            }
            this.fileName = fileName;
            this.lastModified = lastModified;
            this.fileSize = fileSize;
            this.dataOffset = dataOffset;
            this.childNodes = childNodes;
            this.totalChildNoesCount = 0;
        }

        public FileNode(AbstractFile file) throws IOException {
            this(file, file.isDirectory());
        }

        public FileNode(AbstractFile file, boolean isDirectory) throws IOException {
            this(file.getName(), file.lastModified(), file.length(), 0,
                    isDirectory ? new ArrayList<>() : null);
        }

        public FileNode(AbstractFile file, int childCount) throws IOException {
            this(file.getName(), file.lastModified(), file.length(), 0,
                    childCount > 0 ? new ArrayList<>(childCount) : new ArrayList<>());
        }

        public void add(FileNode fileNode) {
            childNodes.add(fileNode);
            this.fileSize += fileNode.fileSize;
            this.totalChildNoesCount += (fileNode.totalChildNoesCount + 1);
        }

        public boolean isDirectory() {
            return null != childNodes;
        }

        public String hashKey() {
            return this.fileSize > 0 ? Hex.encodeHexString(md5sum) + this.fileSize : "";
        }
    }

    public interface EnumCallback {
        void onEnumFile(FileNode fileNode, AbstractFile file) throws IOException;
    }

    private static FileNode recurseEnum(final AbstractFile rootDir, String relativePath, int recurseLevel,
                                        EnumCallback cb) throws IOException {
        AbstractFile curDir;
        if (relativePath.length() > 0) {
            curDir = rootDir.getChildFile(relativePath);
        } else {
            curDir = rootDir;
        }
        FileNode fileNode;
        AbstractFile[] files;
        if (curDir.isDirectory()) {
            files = curDir.listFiles();
            fileNode = new FileNode(curDir, files != null ? files.length : 0);
        } else {
            files = null;
            fileNode = new FileNode(curDir, false);
        }
        fileNode.relativeName = relativePath;
        cb.onEnumFile(fileNode, curDir);
        if (null == files || 0 == files.length) {
            return fileNode;
        }
        Arrays.sort(files, (o1, o2) -> o1.getName().compareToIgnoreCase(o2.getName()));
        for (AbstractFile file : files) {
            String name = file.getName();
            if (name.startsWith(".cgs_vfs_")) {
                continue;
            }
            FileNode childNode;
            String relativeName = relativePath + name;
            if (file.isDirectory()) {
                if (recurseLevel <= 0) {
                    throw new IOException("path too deep");
                }
                childNode = recurseEnum(rootDir, relativeName + "/", recurseLevel - 1, cb);
            } else {
                System.out.println("Add file: " + relativeName + ", Size: " + App.humanReadableBytes(file.length()));
                childNode = new FileNode(file);
                childNode.relativeName = relativeName;
                cb.onEnumFile(childNode, file);
            }
            fileNode.add(childNode);
        }
        return fileNode;
    }

    public static FileNode enumerate(AbstractFile inputDir, EnumCallback cb) throws IOException {
        return recurseEnum(inputDir, "", 260, cb);
    }

    public static class BuildResult {
        public int fileNodesCount;
        public int fileNodesOffset;
        public int checksumsOffset;
    } //class MetadataInfo

    public static class Builder {

        static class Data {
            int size;
            int ptr;

            Data(int s, int p) {
                size = s;
                ptr = p;
            }
        }
        private RandomAccessByteArrayOutputStream output;
        private Map<String, Data> stringPointerMap;
        private List<FileNode> fileNodes;

        public BuildResult build(RandomAccessByteArrayOutputStream os, FileNode rootNode) throws IOException {
            BuildResult result = new BuildResult();
            output = os;
            output.order(ByteOrder.LITTLE_ENDIAN);
            output.seek(0);
            output.setLength(MD_HEADER_SIZE);
            stringPointerMap = new HashMap<>(65536);
            traverse(rootNode);
            result.fileNodesCount = fileNodes.size();
            result.fileNodesOffset = output.getLength();
            result.checksumsOffset = result.fileNodesOffset + MD_NODE_SIZE * fileNodes.size();
            output.setLength(result.checksumsOffset + MD_CHECKSUM_SIZE * fileNodes.size());
            int index = 0;
            for (FileNode fileNode : fileNodes) {
                writeNode(fileNode, result.fileNodesOffset, index++);
            }
            output.setLength(output.getPosition());
            output.seek(0);
            output.writeBytes(MD_FORMAT_MAGIC);
            output.writeByte((byte) MD_MAJOR_VER);
            output.writeByte((byte) MD_MINOR_VER);
            output.writeShort((short) MD_BYTEORDER_FLAG);
            output.writeByte((byte) MD_HEADER_SIZE);
            output.writeByte((byte) MD_NODE_SIZE);
            output.writeInt(result.fileNodesOffset);
            output.writeInt(fileNodes.size());
            output.writeShort((short) 0);
            return result;
        }

        private void writeNode(FileNode fileNode, int offset, int index) {
            String fileNameUtf8 = new String(fileNode.fileName.getBytes(StandardCharsets.UTF_8),
                    StandardCharsets.UTF_8);
            Data data = stringPointerMap.get(fileNameUtf8);
            long lastModified = fileNode.lastModified / 1000;
            output.seek(offset + index * MD_NODE_SIZE);
            output.writeByte((byte) (fileNode.childNodes == null ? 0 : 0x80));
            output.writeByte((byte) data.size);
            output.writeInt(data.ptr);
            output.writeShort((short) ((lastModified & 0xFF00000000L) >> 32));
            output.writeInt((int) lastModified);
            output.writeLong(fileNode.fileSize);
            if (fileNode.childNodes == null) {
                output.writeLong(fileNode.dataOffset);
            } else {
                output.writeInt(fileNode.childNodes.size());
                output.writeInt(offset + fileNode.childIndex * MD_NODE_SIZE);
            }
            if (null == fileNode.childNodes && fileNode.fileSize > 0) {
                App.Assert(null != fileNode.md5sum);
                output.seek(offset + fileNodes.size() * MD_NODE_SIZE + index * MD_CHECKSUM_SIZE);
                output.write(fileNode.md5sum, 0, MD_CHECKSUM_SIZE);
            }
        }

        @SuppressWarnings("all")
        private void traverse(FileNode rootNode) throws IOException {
            ArrayList<byte[]> strings = new ArrayList<>(65536);
            fileNodes = new ArrayList<FileNode>(128 * 1024);
            fileNodes.add(rootNode);
            traverse(strings, rootNode);
            strings.sort(new Comparator<byte[]>() {
                @Override
                public int compare(final byte[] o1, final byte[] o2) {
                    return Integer.compare(o2.length, o1.length);
                }
            });
            int[] offsets = new int[strings.size()];
            for (int i = 0; i < strings.size(); ++i) {
                byte[] curr = strings.get(i);
                int j = i - 1;
                while (j >= 0) {
                    final byte[] find = strings.get(j);
                    int n = indexOf(curr, find);
                    if (n != -1) {
                        int ptr = offsets[j] + n;
                        offsets[i] = ptr;
                        stringPointerMap.put(new String(curr, StandardCharsets.UTF_8), new Data(curr.length, ptr));
                        break;
                    }
                    --j;
                }
                if (j < 0) {
                    int ptr = output.getLength();
                    offsets[i] = ptr;
                    output.seek(ptr);
                    output.write(curr);
                    stringPointerMap.put(new String(curr, StandardCharsets.UTF_8), new Data(curr.length, ptr));
                }
            }
        }

        private void traverse(ArrayList<byte[]> strings, final FileNode fileNode) throws IOException {
            byte[] fileNameBytes = fileNode.fileName.getBytes(StandardCharsets.UTF_8);
            if (fileNameBytes.length > 255) {
                throw new IOException("File name too long: " + fileNode.fileName);
            }
            strings.add(fileNameBytes);
            if (fileNode.childNodes != null) {
                fileNode.childIndex = fileNodes.size();
                fileNodes.addAll(fileNode.childNodes);
                for (FileNode sunFileNode : fileNode.childNodes) {
                    traverse(strings, sunFileNode);
                }
            }
        }

        private static int indexOf(byte[] a, byte[] b) {
            for (int i = 0; i <= a.length - b.length; ++i) {
                int j = 0;
                while (j < b.length) {
                    if (a[i + j] != b[j]) {
                        break;
                    }
                    ++j;
                }
                if (j == b.length) {
                    return i;
                }
            }
            return -1;
        }
    } // class Builder

    public static class Loader {

        final ByteBuffer data;
        final int headerSize;
        final int nodeSize;
        final int nodesOffset;
        final int nodesCount;
        final int md5Offset;

        public Loader(RandomAccessDataInput in, long offset, int len) throws IOException {
            this(in.readBytes(offset, len));
        }

        public Loader(byte[] metadata) throws IOException {
            this(metadata, 0, metadata.length);
        }

        public Loader(byte[] metadata, int off, int len) throws IOException {
            this.data = ByteBuffer.wrap(metadata, off, len);
            byte[] magic = new byte[4];
            data.get(magic, 0, magic.length);
            if (!MD_FORMAT_MAGIC.equals(new String(magic)) || MD_MAJOR_VER != data.get() ||
                    MD_MINOR_VER_INVALID == data.get()) {
                throw new IOException("file metadata format");
            } else if (0 != data.get()) { // ByteOrder byte1
                data.order(ByteOrder.LITTLE_ENDIAN);
            }
            data.get(); // ByteOrder byte2
            this.headerSize = data.get();
            this.nodeSize = data.get();
            this.nodesOffset = data.getInt();
            this.nodesCount = data.getInt();
            this.md5Offset = nodesOffset + (nodeSize * nodesCount);
            assert headerSize >= MD_HEADER_SIZE;
            if (md5Offset + 16 * nodesCount > data.limit()) {
                throw new IOException("size");
            }
        }

        private FileNode load(int offset) throws IOException {
            FileNode fileNode;
            data.position(offset);
            final int flags = data.get();
            final boolean isDirectory = (0 != (flags & 0x80));
            final int nameLength = 0xff & data.get();
            final int namePosition = data.getInt();
            final long lastModifiedHigh = (0x00000000FFFFFFFFL & data.getShort());
            final long lastModifiedLow = (0x00000000FFFFFFFFL & data.getInt());
            final long lastModified = ((lastModifiedHigh << 32) | lastModifiedLow) * 1000;
            final long fileSize = data.getLong();
            long dataOffset = 0;
            int childNodesCount = 0;
            int childNodesPosition = 0;
            if (isDirectory) {
                childNodesCount = data.getInt();
                childNodesPosition = data.getInt();
            } else {
                dataOffset = data.getLong();
            }
            byte[] nameBytes = new byte[nameLength];
            data.position(namePosition);
            data.get(nameBytes);
            String name = new String(nameBytes, StandardCharsets.UTF_8);
            fileNode = new FileNode(name, lastModified, fileSize, dataOffset,
                    isDirectory ? new ArrayList<>(childNodesCount) : null);
            if (isDirectory) {
                offset = childNodesPosition;
                for (int i = 0; i < childNodesCount; ++i, offset += nodeSize) {
                    FileNode childNode = load(offset);
                    fileNode.childNodes.add(childNode);
                }
                Collections.sort(fileNode.childNodes);
            } else {
                fileNode.md5sum = new byte[16];
                int index = (offset - nodesOffset) / nodeSize;
                data.position(md5Offset + 16 * index);
                data.get(fileNode.md5sum);
            }
            return fileNode;
        }

        public FileNode load() throws IOException {
            return load(nodesOffset);
        }
    } // class Loader

    public static FileNode load(RandomAccessDataInput in, long off, int len) throws IOException {
        return new Loader(in, off, len).load();
    }

    public static FileNode load(byte[] md, int off, int len) throws IOException {
        return new Loader(md, off, len).load();
    }

    public static FileNode load(byte[] md) throws IOException {
        return new Loader(md, 0, md.length).load();
    }

    public static void check(byte[] data) {
        ByteBuffer buff = ByteBuffer.wrap(data);
        buff.order(ByteOrder.LITTLE_ENDIAN);
        System.out.println("Magic: " + Integer.toHexString(buff.getInt()));
        System.out.println("MajorVer: " + buff.get());
        System.out.println("MinorVer: " + buff.get());
        System.out.println("ByteOrder: " + buff.getShort());
        final int headerSize = buff.get();
        final int nodeSize = buff.get();
        final int rootPtr = buff.getInt();
        System.out.println("HeaderSize: " + headerSize);
        System.out.println("NodeSize: " + nodeSize);
        System.out.println("RootPtr: " + rootPtr);
        checkNode(data, rootPtr, nodeSize);
    }

    private static void checkNode(final byte[] data, final int ptr, final int nodeSize) {
        ByteBuffer node = ByteBuffer.wrap(data, ptr, nodeSize);
        node.order(ByteOrder.LITTLE_ENDIAN);
        System.out.println();
        boolean isDir = (node.get() & 0x80) != 0;
        System.out.println("Type: " + (isDir ? "dir" : "file"));
        int nameLength = node.get();
        int namePtr = node.getInt();
        System.out.println("Name: " + new String(data, namePtr, nameLength, StandardCharsets.UTF_8));
        long lastModifiedHigh = node.getShort();
        long lastModifiedLow = node.getInt();
        long lastModified = ((lastModifiedHigh << 32) | lastModifiedLow) * 1000;
        System.out.println("Last-Modified: "
                + new SimpleDateFormat("yyyy-MM-dd HH:mm:ss").format(new Date(lastModified)));
        System.out.println("FileSize: " + node.getLong());
        if (isDir) {
            final int nodesCount = node.getInt();
            int subNodesPtr = node.getInt();
            System.out.println("SubFileCount: " + nodesCount);
            for (int i = 0; i < nodesCount; ++i) {
                checkNode(data, subNodesPtr, nodeSize);
                subNodesPtr += nodeSize;
            }
        } else {
            System.out.println("EntryOffset: " + node.getLong());
        }
    }
}
