package com.tencent.start.cgs.tools;

import com.emc.ecs.nfsclient.nfs.NfsReadResponse;
import com.emc.ecs.nfsclient.nfs.io.Nfs3File;

import java.io.*;

@SuppressWarnings("unused")
public abstract class RandomAccessInputFile extends RandomAccessDataInput {

    public abstract long length() throws IOException;
    public abstract long getFilePointer() throws IOException;
    @Override
    public int read(byte[] buf, int off, int len) throws IOException {
        if (off < 0 || len < 0) {
            throw new IndexOutOfBoundsException();
        }
        final long maxLen = length();
        long pos = getFilePointer();
        if (pos >= maxLen) {
            return -1;
        } else if (pos + len > maxLen) {
            len = (int) (maxLen - pos);
        }
        if (len > 0) {
            readFully(buf, off, len);
        }
        return len;
    }

    public abstract int pread(byte[] buf, int off, int len, long position) throws IOException;

    @Override
    public int available() throws IOException {
        final long maxLen = length();
        long pos = getFilePointer();
        if (pos >= maxLen) {
            return 0;
        }
        long len = maxLen - pos;
        return (int) Math.min(len, Integer.MAX_VALUE);
    }

    public static class RandomAccessNativeInputFile extends RandomAccessInputFile {

        protected final RandomAccessFile file;

        public RandomAccessNativeInputFile(File file) throws IOException {
            this.file = new RandomAccessFile(file, "r");
        }

        public RandomAccessNativeInputFile(RandomAccessFile file) {
            this.file = file;
        }

        @Override
        public void close() throws IOException {
            file.close();
        }

        @Override
        public long length() throws IOException {
            return file.length();
        }

        @Override
        public long getFilePointer() throws IOException {
            return file.getFilePointer();
        }

        public void seek(long pos) throws IOException {
            file.seek(pos);
        }

        @Override
        public int read() throws IOException {
            return file.read();
        }

        @Override
        public void readFully(byte[] buf, int off, int len) throws IOException {
            file.readFully(buf, off, len);
        }

        @Override
        public int skipBytes(int n) throws IOException {
            return file.skipBytes(n);
        }

        @Override
        public int pread(byte[] buf, int off, int len, long position) throws IOException {
            seek(position);
            return read(buf, off, len);
        }
    }

    public static RandomAccessInputFile from(RandomAccessFile file) {
        return new RandomAccessNativeInputFile(file);
    }

    public static RandomAccessInputFile from(File file) throws IOException {
        return new RandomAccessNativeInputFile(new RandomAccessFile(file, "r"));
    }

    public static class RandomAccessNfsInputFile extends RandomAccessInputFile {

        protected final Nfs3File file;
        protected final int rtmax;
        protected final byte[] buf;
        protected long filePos;

        public RandomAccessNfsInputFile(Nfs3File file) throws IOException {
            this.file = file;
            this.rtmax = (int) Math.min(file.fsinfo().getFsInfo().rtmax, Integer.MAX_VALUE);
            this.buf = new byte[1];
            this.filePos = 0;
        }

        @Override
        public void close() throws IOException {
        }

        @Override
        public long length() throws IOException {
            return file.length();
        }

        @Override
        public long getFilePointer() throws IOException {
            return filePos;
        }

        public void seek(long pos) throws IOException {
            if (pos < 0) {
                throw new IOException("Negative seek offset");
            } else if (pos != getFilePointer()) {
                filePos = pos;
            }
        }

        @Override
        public int read() throws IOException {
            readFully(buf, 0, 1);
            return buf[0];
        }

        @Override
        public void readFully(byte[] b, int off, int len) throws IOException {
            if (null == b) {
                throw new NullPointerException();
            } else if (0 == len) {
                return;
            }
            int bytesRead = 0;
            for (;;) {
                int bytesToRead = len - bytesRead;
                if (bytesToRead > rtmax) {
                    bytesToRead = rtmax;
                }
                NfsReadResponse resp = file.read(filePos, bytesToRead, b, off + bytesRead);
                int bytes = resp.getBytesRead();
                if ((bytesRead += bytes) >= len) {
                    return;
                } else if (resp.isEof()) {
                    throw new EOFException();
                }
            }
        }

        @Override
        public int skipBytes(int n) throws IOException {
            if (filePos + n < file.length()) {
                filePos += n;
                return n;
            } else if (filePos < file.length()) {
                n = (int) (file.length() - filePos);
                filePos = file.length();
                return n;
            }
            return 0;
        }

        @Override
        public int pread(byte[] buf, int off, int len, long position) throws IOException {
            if (position < 0) {
                throw new IOException("Negative seek offset");
            }
            int bytesRead = 0;
            for (;;) {
                int bytesToRead = len - bytesRead;
                if (bytesToRead > rtmax) {
                    bytesToRead = rtmax;
                }
                NfsReadResponse resp = file.read(position + bytesRead, bytesToRead, buf, off + bytesRead);
                int bytes = resp.getBytesRead();
                if ((bytesRead += bytes) >= len || resp.isEof()) {
                    break;
                }
            }
            return bytesRead;
        }
    }

    public static RandomAccessInputFile from(Nfs3File file) throws IOException {
        return new RandomAccessNfsInputFile(file);
    }

    public static RandomAccessInputFile from(AbstractFile file) throws IOException {
        if (file.getClassType().equals(Nfs3File.class)) {
            return new RandomAccessNfsInputFile((Nfs3File) file.asFile());
        } else {
            return new RandomAccessNativeInputFile((File) file.asFile());
        }
    }
}
